#!/bin/bash

################################################################################
# Consul Backup Script
# Creates a snapshot of Consul cluster state via HTTP API
#
# Usage: ./consul-backup.sh
################################################################################

set -e
set -o pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_status() { echo -e "${GREEN}[✓]${NC} $1"; }
print_error() { echo -e "${RED}[✗]${NC} $1"; }
print_info() { echo -e "${BLUE}[i]${NC} $1"; }

CONSUL_ENDPOINT="http://gv-essentials-consul-server:8500"
BACKUP_IMAGE="docker.io/getvisibility/backup:0.1.4"

main() {
    echo ""
    echo "========================================"
    echo "Consul Backup Script"
    echo "========================================"
    echo ""

    BACKUP_DATE=$(date +%Y%m%d-%H%M%S)
    BACKUP_FILE="./consul-backup-${BACKUP_DATE}.snap"
    POD_NAME="consul-backup-$$"

    # Start backup pod
    print_info "Starting backup pod..."
    if ! kubectl run "${POD_NAME}" --image="${BACKUP_IMAGE}" --restart=Never --command -- sleep 3600 >/dev/null 2>&1; then
        print_error "Failed to start backup pod"
        exit 1
    fi

    # Wait for pod to be ready
    for i in $(seq 1 30); do
        if kubectl get pod "${POD_NAME}" -o jsonpath='{.status.phase}' 2>/dev/null | grep -q "Running"; then
            break
        fi
        sleep 1
    done

    # Cleanup function
    cleanup() {
        kubectl delete pod "${POD_NAME}" --force --grace-period=0 >/dev/null 2>&1 || true
    }
    trap cleanup EXIT INT TERM

    print_status "Backup pod ready"

    # Test Consul endpoint
    print_info "Testing Consul endpoint..."
    if ! kubectl exec -i "${POD_NAME}" -- curl -sf "${CONSUL_ENDPOINT}/v1/status/leader" >/dev/null 2>&1; then
        print_error "Cannot connect to Consul at ${CONSUL_ENDPOINT}"
        exit 1
    fi
    print_status "Connected to Consul"

    # Create snapshot via HTTP API
    print_info "Creating Consul snapshot..."
    if ! kubectl exec -i "${POD_NAME}" -- curl -sf "${CONSUL_ENDPOINT}/v1/snapshot" -o /tmp/snapshot.snap 2>/dev/null; then
        print_error "Failed to create snapshot"
        exit 1
    fi

    # Copy snapshot from pod
    print_info "Copying snapshot from pod..."
    if ! kubectl cp "${POD_NAME}:/tmp/snapshot.snap" "${BACKUP_FILE}" >/dev/null 2>&1; then
        print_error "Failed to copy snapshot from pod"
        exit 1
    fi

    # Verify
    if [ ! -s "$BACKUP_FILE" ]; then
        print_error "Backup file not created or empty"
        exit 1
    fi

    BACKUP_SIZE=$(du -h "$BACKUP_FILE" | awk '{print $1}')
    print_status "Snapshot created: $BACKUP_SIZE"

    echo ""
    echo "========================================"
    echo "Backup Complete"
    echo "========================================"
    echo "File: $BACKUP_FILE"
    echo "Size: $BACKUP_SIZE"
}

main "$@"

