#!/bin/bash

################################################################################
# Consul Restore Script
# Restores Consul cluster state from a snapshot via HTTP API
#
# Usage: ./consul-restore.sh <consul-backup-YYYYMMDD-HHMMSS.snap> [--yes]
################################################################################

set -e
set -o pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_status() { echo -e "${GREEN}[✓]${NC} $1"; }
print_error() { echo -e "${RED}[✗]${NC} $1"; }
print_info() { echo -e "${BLUE}[i]${NC} $1"; }

CONSUL_ENDPOINT="http://gv-essentials-consul-server:8500"
BACKUP_IMAGE="docker.io/getvisibility/backup:0.1.4"

main() {
    echo ""
    echo "========================================"
    echo "Consul Restore Script"
    echo "========================================"
    echo ""

    # Args and confirmation
    if [ $# -lt 1 ]; then
        echo -e "${RED}Usage:${NC} $0 <consul-backup-YYYYMMDD-HHMMSS.snap> [--yes]"
        exit 1
    fi

    SNAPSHOT_FILE="$1"; shift || true
    YES_FLAG=0
    for arg in "$@"; do
        if [ "$arg" = "--yes" ]; then YES_FLAG=1; fi
    done

    if [ ! -f "${SNAPSHOT_FILE}" ] || [[ "${SNAPSHOT_FILE}" != *.snap ]]; then
        print_error "Snapshot not found or not a .snap file: ${SNAPSHOT_FILE}"
        exit 1
    fi

    echo -e "${YELLOW}WARNING:${NC} This restore will overwrite current Consul state.\n\
- Service registrations will be replaced\n\
- Key/value store will be replaced\n\
- ACLs and configurations will be replaced\n\
- Proceed only if you understand the risk\n"

    if [ $YES_FLAG -ne 1 ]; then
        printf "Type 'yes' to continue: "
        read -r CONFIRM
        if [ "${CONFIRM}" != "yes" ]; then
            echo "Aborted."
            exit 1
        fi
    fi

    POD_NAME="consul-restore-$$"

    # Start backup pod
    print_info "Starting restore pod..."
    if ! kubectl run "${POD_NAME}" --image="${BACKUP_IMAGE}" --restart=Never --command -- sleep 3600 >/dev/null 2>&1; then
        print_error "Failed to start restore pod"
        exit 1
    fi

    # Wait for pod to be ready
    for i in $(seq 1 30); do
        if kubectl get pod "${POD_NAME}" -o jsonpath='{.status.phase}' 2>/dev/null | grep -q "Running"; then
            break
        fi
        sleep 1
    done

    # Cleanup function
    cleanup() {
        kubectl delete pod "${POD_NAME}" --force --grace-period=0 >/dev/null 2>&1 || true
    }
    trap cleanup EXIT INT TERM

    print_status "Restore pod ready"

    # Test Consul endpoint
    print_info "Testing Consul endpoint..."
    if ! kubectl exec -i "${POD_NAME}" -- curl -sf "${CONSUL_ENDPOINT}/v1/status/leader" >/dev/null 2>&1; then
        print_error "Cannot connect to Consul at ${CONSUL_ENDPOINT}"
        exit 1
    fi
    print_status "Connected to Consul"

    # Copy snapshot to pod
    print_info "Copying snapshot to pod..."
    if ! kubectl cp "${SNAPSHOT_FILE}" "${POD_NAME}:/tmp/restore.snap" >/dev/null 2>&1; then
        print_error "Failed to copy snapshot to pod"
        exit 1
    fi
    print_status "Snapshot copied"

    # Restore snapshot via HTTP API
    print_info "Restoring Consul snapshot..."
    HTTP_CODE=$(kubectl exec -i "${POD_NAME}" -- curl -sf -w "%{http_code}" -X PUT --data-binary @/tmp/restore.snap "${CONSUL_ENDPOINT}/v1/snapshot" -o /dev/null 2>/dev/null || echo "000")
    
    if [ "$HTTP_CODE" = "200" ]; then
        print_status "Restore completed (HTTP 200)"
    else
        print_error "Restore failed (HTTP ${HTTP_CODE})"
        exit 1
    fi

    # Verify cluster health
    print_info "Verifying cluster health..."
    if kubectl exec -i "${POD_NAME}" -- curl -sf "${CONSUL_ENDPOINT}/v1/status/leader" >/dev/null 2>&1; then
        print_status "Cluster healthy"
    else
        print_error "Cluster health check failed"
        exit 1
    fi

    echo ""
    echo "========================================"
    echo "Restore Complete"
    echo "========================================"
    echo "File: ${SNAPSHOT_FILE}"
}

main "$@"

