#!/bin/bash

set -e

set -o pipefail

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
NC='\033[0m'

echo "========================================"
echo "Elasticsearch Backup - Elasticdump"
echo "========================================"

# Get password
export ES_PASSWORD=$(kubectl get secrets/gv-es-elastic-user -o go-template='{{.data.elastic | base64decode}}{{"\n"}}')

# Prepare archive target and temp work dir
BACKUP_DATE=$(date +%Y%m%d-%H%M%S)
ARCHIVE_FILE="./elasticsearch-backup-${BACKUP_DATE}.tar.gz"
WORK_DIR="$(mktemp -d -t es-backup-${BACKUP_DATE}-XXXXXX)"
LOG_FILE="./elasticsearch-backup-${BACKUP_DATE}.log"

echo "Archive: ${ARCHIVE_FILE}"

# Start port-forward if needed and ensure cleanup
STARTED_PF=0
PF_PID=""
cleanup() {
    if [ "$STARTED_PF" -eq 1 ] && [ -n "$PF_PID" ]; then
        kill "$PF_PID" >/dev/null 2>&1 || true
    fi
}
trap cleanup EXIT INT TERM

ensure_connection() {
    if curl -s -u elastic:${ES_PASSWORD} http://localhost:9200/_cluster/health > /dev/null 2>&1; then
        return 0
    fi
    kubectl port-forward svc/gv-es-http 9200:9200 >/dev/null 2>&1 &
    PF_PID=$!
    STARTED_PF=1
    for _ in $(seq 1 30); do
        if curl -s -u elastic:${ES_PASSWORD} http://localhost:9200/_cluster/health > /dev/null 2>&1; then
            return 0
        fi
        sleep 1
    done
    echo -e "${RED}ERROR: Cannot connect to Elasticsearch${NC}"
    return 1
}

# Test/establish connection
echo -n "Checking connection... "
if ensure_connection; then
    echo -e "${GREEN}OK${NC}"
else
    rm -rf "${WORK_DIR}"
    exit 1
fi

# Run backup (quiet; logs saved to file)
echo "Starting backup..."
if ! multielasticdump \
  --direction=dump \
  --match='^(?!\.).*' \
  --input=http://elastic:${ES_PASSWORD}@localhost:9200 \
  --output="${WORK_DIR}" \
  --concurrency=5 \
  --limit=5000 \
  --quiet > "${LOG_FILE}" 2>&1; then
  echo -e "${RED}ERROR: Backup failed${NC}"
  echo "See log: ${LOG_FILE}"
  rm -rf "${WORK_DIR}"
  exit 1
fi

# Count dumped indices/files before archiving
COUNT=$(find "${WORK_DIR}" -maxdepth 1 -type f -name '*.json' | wc -l | awk '{print $1}')

# Create archive
tar -czf "${ARCHIVE_FILE}" -C "${WORK_DIR}" .
[ ! -s "${ARCHIVE_FILE}" ] && { echo -e "${RED}ERROR: Failed to create archive${NC}"; rm -rf "${WORK_DIR}"; exit 1; }

# Cleanup work dir
rm -rf "${WORK_DIR}"

# Verify archive
SIZE=$(du -sh "${ARCHIVE_FILE}" | awk '{print $1}')

echo ""
echo -e "${GREEN}✓ Backup completed${NC}"
echo "  Size: ${SIZE}"
echo "  Indices: ${COUNT}"
echo "  Location: ${ARCHIVE_FILE}"
echo "  Log: ${LOG_FILE}"