#!/bin/bash

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

echo "========================================"
echo "Elasticsearch Restore - Elasticdump"
echo "========================================"

# Args and confirmation
if [ $# -lt 1 ]; then
    echo -e "${RED}Usage:${NC} $0 <elasticsearch-backup-YYYYMMDD-HHMMSS.tar.gz> [--yes]"
    exit 1
fi

ARCHIVE_FILE="$1"
shift || true

YES_FLAG=0
for arg in "$@"; do
    if [ "$arg" = "--yes" ]; then YES_FLAG=1; fi
done

if [ ! -f "${ARCHIVE_FILE}" ] || [[ "${ARCHIVE_FILE}" != *.tar.gz ]]; then
    echo -e "${RED}ERROR:${NC} Archive not found or not a .tar.gz: ${ARCHIVE_FILE}"
    exit 1
fi

echo -e "${YELLOW}WARNING:${NC} This restore will overwrite indices in the target cluster.\n\
- Indices may be deleted and recreated\n\
- Mappings and settings will be applied from the backup\n\
- Any divergent data may be lost\n"

if [ $YES_FLAG -ne 1 ]; then
    printf "Type 'yes' to continue: "
    read -r CONFIRM
    if [ "${CONFIRM}" != "yes" ]; then
        echo "Aborted."
        exit 1
    fi
fi

# Extract archive into temp directory
EXTRACT_DIR=$(mktemp -d -t es-restore-XXXXXX)
BACKUP_DIR="${EXTRACT_DIR}"
tar -xf "${ARCHIVE_FILE}" -C "${EXTRACT_DIR}"

echo "Backup: ${ARCHIVE_FILE}"

# Get password
export ES_PASSWORD=$(kubectl get secrets/gv-es-elastic-user -o go-template='{{.data.elastic | base64decode}}{{"\n"}}')

# Start port-forward if needed and ensure cleanup
STARTED_PF=0
PF_PID=""
cleanup() {
    if [ "$STARTED_PF" -eq 1 ] && [ -n "$PF_PID" ]; then
        kill "$PF_PID" >/dev/null 2>&1 || true
    fi
    if [ -n "$EXTRACT_DIR" ]; then
        rm -rf "$EXTRACT_DIR" >/dev/null 2>&1 || true
    fi
}
trap cleanup EXIT INT TERM

ensure_connection() {
    if curl -s -u elastic:${ES_PASSWORD} http://localhost:9200/_cluster/health > /dev/null 2>&1; then
        return 0
    fi
    kubectl port-forward svc/gv-es-http 9200:9200 >/dev/null 2>&1 &
    PF_PID=$!
    STARTED_PF=1
    for _ in $(seq 1 30); do
        if curl -s -u elastic:${ES_PASSWORD} http://localhost:9200/_cluster/health > /dev/null 2>&1; then
            return 0
        fi
        sleep 1
    done
    echo -e "${RED}ERROR: Cannot connect to Elasticsearch${NC}"
    return 1
}

# Test/establish connection
echo -n "Checking connection... "
if ensure_connection; then
    echo -e "${GREEN}OK${NC}"
else
    exit 1
fi

# Get current state
BEFORE=$(curl -s -u elastic:${ES_PASSWORD} http://localhost:9200/_cat/indices?h=index | grep -v '^\.' | wc -l)
BEFORE=$(echo "$BEFORE" | awk '{print $1}')
echo "Indices before restore: ${BEFORE}"

# Restore
echo ""
echo -e "${YELLOW}Starting restore...${NC}"
multielasticdump \
  --direction=load \
  --match='^(?!\.).*' \
  --input="${BACKUP_DIR}" \
  --output=http://elastic:${ES_PASSWORD}@localhost:9200 \
  --concurrency=3 \
  --deleteIndex=true \
  --limit=5000

# Wait for settlement
sleep 30

# Get new state
AFTER=$(curl -s -u elastic:${ES_PASSWORD} http://localhost:9200/_cat/indices?h=index | grep -v '^\.' | wc -l)
AFTER=$(echo "$AFTER" | awk '{print $1}')
HEALTH=$(curl -s -u elastic:${ES_PASSWORD} http://localhost:9200/_cluster/health | jq -r '.status')

echo ""
echo -e "${GREEN}✓ Restore completed${NC}"
echo "  Indices before: ${BEFORE}"
echo "  Indices after: ${AFTER}"
echo "  Cluster health: ${HEALTH}"

# Verify
if [ "${AFTER}" -ge "${BEFORE}" ] && [ "${HEALTH}" != "red" ]; then
    echo -e "${GREEN}✓ Restoration verified${NC}"
else
    echo -e "${RED}✗ Verification failed - Check cluster state${NC}"
    exit 1
fi