#!/bin/bash

################################################################################
# PostgreSQL Backup Script - WITH TIMEOUT AND ERROR HANDLING
# Direct backup via kubectl exec with timeout protection
#
# Usage: ./backup-postgresql.sh
################################################################################

set -e

set -o pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_status() { echo -e "${GREEN}[✓]${NC} $1"; }
print_error() { echo -e "${RED}[✗]${NC} $1"; }
print_info() { echo -e "${BLUE}[i]${NC} $1"; }

main() {
    echo ""
    echo "========================================"
    echo "PostgreSQL Backup Script"
    echo "========================================"
    echo ""
    
    # Get PostgreSQL password
    print_info "Retrieving PostgreSQL password..."
    PG_PASSWORD=$(kubectl get secret gv.gv-postgresql.credentials.postgresql.acid.zalan.do -o go-template='{{.data.password | base64decode}}' 2>/dev/null)
    [ -z "$PG_PASSWORD" ] && { print_error "Failed to get password"; exit 1; }
    print_status "Retrieved"
    
    # Find PostgreSQL pod
    print_info "Finding PostgreSQL pod..."
    PG_NS_POD=$(kubectl get pods --all-namespaces -o wide 2>/dev/null | awk 'tolower($2) ~ /postgresql/ {print $1" "$2; exit}')
    PG_NAMESPACE=$(echo "$PG_NS_POD" | awk '{print $1}')
    PG_POD=$(echo "$PG_NS_POD" | awk '{print $2}')
    [ -z "$PG_NAMESPACE" -o -z "$PG_POD" ] && { print_error "PostgreSQL pod not found"; exit 1; }
    print_status "Found: $PG_POD (namespace: $PG_NAMESPACE)"

    # Fallback: retrieve password in pod namespace if not found earlier
    if [ -z "$PG_PASSWORD" ]; then
        print_info "Retrying password retrieval in namespace $PG_NAMESPACE..."
        PG_PASSWORD=$(kubectl -n "$PG_NAMESPACE" get secret gv.gv-postgresql.credentials.postgresql.acid.zalan.do -o go-template='{{.data.password | base64decode}}' 2>/dev/null)
        [ -z "$PG_PASSWORD" ] && { print_error "Failed to get password"; exit 1; }
        print_status "Retrieved"
    fi
    
    # Test connection
    print_info "Testing PostgreSQL connection..."
    if ! timeout 10 bash -c "echo 'SELECT 1;' | PGPASSWORD='${PG_PASSWORD}' kubectl -n '${PG_NAMESPACE}' exec -i '${PG_POD}' -- psql -U gv -p 5432 -d postgres" > /dev/null 2>&1; then
        print_error "Cannot connect to PostgreSQL in pod"
        print_info "Trying with different connection method..."
        exit 1
    fi
    print_status "Connected"
    
    BACKUP_DATE=$(date +%Y%m%d-%H%M%S)
    
    # Create a full-cluster dump directly (no discovery step)
    print_info "Creating full SQL dump (entire cluster)..."
    BACKUP_FILE="./postgres-backup-${BACKUP_DATE}.sql.gz"

    # Optional timeouts (opt-in via env)
    PGOPTS=""
    [ -n "${STATEMENT_TIMEOUT_MS}" ] && PGOPTS="${PGOPTS} -c statement_timeout=${STATEMENT_TIMEOUT_MS}"
    [ -n "${LOCK_TIMEOUT_MS}" ] && PGOPTS="${PGOPTS} -c lock_timeout=${LOCK_TIMEOUT_MS}"

    KUBE_CMD="export PGPASSWORD='${PG_PASSWORD}';"
    if [ -n "${PGOPTS}" ]; then
        KUBE_CMD="${KUBE_CMD} export PGOPTIONS='${PGOPTS}';"
    fi
    KUBE_CMD="${KUBE_CMD} pg_dumpall -U gv -p 5432"

    if [ -n "${DUMP_TIMEOUT}" ] && [ "${DUMP_TIMEOUT}" != "0" ]; then
        if timeout "${DUMP_TIMEOUT}" kubectl -n "${PG_NAMESPACE}" --request-timeout=0 exec "${PG_POD}" -- sh -lc "${KUBE_CMD}" | gzip -c > "${BACKUP_FILE}"; then
            :
        else
            print_error "Cluster dump failed or timed out"
            exit 1
        fi
    else
        if kubectl -n "${PG_NAMESPACE}" --request-timeout=0 exec "${PG_POD}" -- sh -lc "${KUBE_CMD}" | gzip -c > "${BACKUP_FILE}"; then
            :
        else
            print_error "Cluster dump failed"
            exit 1
        fi
    fi

    if [ ! -s "$BACKUP_FILE" ]; then
        print_error "Backup file not created or empty"
        exit 1
    fi

    BACKUP_SIZE=$(du -h "$BACKUP_FILE" | awk '{print $1}')
    print_status "Archive created: $BACKUP_SIZE"

    echo ""
    echo "========================================"
    echo "Backup Complete"
    echo "========================================"
    echo "File: $BACKUP_FILE"
    echo "Size: $BACKUP_SIZE"
}

main "$@"
