#!/bin/bash

################################################################################
# PostgreSQL Restore Script - WITH CONFIRMATION AND TIMEOUTS
# Restores a full-cluster pg_dumpall SQL gzip archive into the cluster
#
# Usage: ./pgsql-restore.sh <postgres-backup-YYYYMMDD-HHMMSS.sql.gz> [--yes]
################################################################################

set -e
set -o pipefail

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_status() { echo -e "${GREEN}[✓]${NC} $1"; }
print_error() { echo -e "${RED}[✗]${NC} $1"; }
print_info() { echo -e "${BLUE}[i]${NC} $1"; }

main() {
    echo ""
    echo "========================================"
    echo "PostgreSQL Restore Script"
    echo "========================================"
    echo ""

    # Args and confirmation
    if [ $# -lt 1 ]; then
        echo -e "${RED}Usage:${NC} $0 <postgres-backup-YYYYMMDD-HHMMSS.sql.gz> [--yes]"
        exit 1
    fi

    ARCHIVE_FILE="$1"; shift || true
    YES_FLAG=0
    for arg in "$@"; do
        if [ "$arg" = "--yes" ]; then YES_FLAG=1; fi
    done

    if [ ! -f "${ARCHIVE_FILE}" ] || [[ "${ARCHIVE_FILE}" != *.sql.gz ]]; then
        print_error "Archive not found or not a .sql.gz: ${ARCHIVE_FILE}"
        exit 1
    fi

    echo -e "${YELLOW}WARNING:${NC} This restore will DROP and recreate roles, databases, and objects.\n\
- Existing data may be overwritten\n\
- Mappings and permissions will be replaced\n\
- Proceed only if you understand the risk\n"

    if [ $YES_FLAG -ne 1 ]; then
        printf "Type 'yes' to continue: "
        read -r CONFIRM
        if [ "${CONFIRM}" != "yes" ]; then
            echo "Aborted."
            exit 1
        fi
    fi

    # Get PostgreSQL password (for gv)
    print_info "Retrieving PostgreSQL password..."
    PG_PASSWORD=$(kubectl get secret gv.gv-postgresql.credentials.postgresql.acid.zalan.do -o go-template='{{.data.password | base64decode}}' 2>/dev/null)
    [ -z "$PG_PASSWORD" ] && { print_error "Failed to get password"; exit 1; }
    print_status "Retrieved"

    # Find PostgreSQL pod and namespace
    print_info "Finding PostgreSQL pod..."
    PG_NS_POD=$(kubectl get pods --all-namespaces -o wide 2>/dev/null | awk 'tolower($2) ~ /postgresql/ {print $1" "$2; exit}')
    PG_NAMESPACE=$(echo "$PG_NS_POD" | awk '{print $1}')
    PG_POD=$(echo "$PG_NS_POD" | awk '{print $2}')
    [ -z "$PG_NAMESPACE" -o -z "$PG_POD" ] && { print_error "PostgreSQL pod not found"; exit 1; }
    print_status "Found: $PG_POD (namespace: $PG_NAMESPACE)"

    # Choose DB user: prefer superuser 'postgres' if available
    DB_USER="gv"
    AUTH_EXPORT="export PGPASSWORD='${PG_PASSWORD}';"
    if kubectl -n "${PG_NAMESPACE}" exec -i "${PG_POD}" -- sh -lc "psql -U postgres -p 5432 -d postgres -t -A -c 'SELECT 1'" >/dev/null 2>&1; then
        DB_USER="postgres"
        AUTH_EXPORT=""
        print_status "Using superuser: postgres"
    else
        print_status "Using user: gv"
    fi

    # Test connection
    print_info "Testing PostgreSQL connection..."
    if ! kubectl -n "${PG_NAMESPACE}" exec -i "${PG_POD}" -- sh -lc "${AUTH_EXPORT} psql -U ${DB_USER} -p 5432 -d postgres -t -A -c 'SELECT 1'" >/dev/null 2>&1; then
        print_error "Cannot connect to PostgreSQL in pod as ${DB_USER}"
        exit 1
    fi
    print_status "Connected"

    # No pre-clean needed - dump is idempotent without --clean

    # Restore
    print_info "Starting restore from ${ARCHIVE_FILE}..."

    # Optional timeouts (opt-in via env)
    PGOPTS=""
    [ -n "${STATEMENT_TIMEOUT_MS}" ] && PGOPTS="${PGOPTS} -c statement_timeout=${STATEMENT_TIMEOUT_MS}"
    [ -n "${LOCK_TIMEOUT_MS}" ] && PGOPTS="${PGOPTS} -c lock_timeout=${LOCK_TIMEOUT_MS}"

    KUBE_CMD="${AUTH_EXPORT}"
    if [ -n "${PGOPTS}" ]; then
        KUBE_CMD="${KUBE_CMD} export PGOPTIONS='${PGOPTS}';"
    fi
    KUBE_CMD="${KUBE_CMD} psql -U ${DB_USER} -p 5432 -d postgres"

    if [ -n "${RESTORE_TIMEOUT}" ] && [ "${RESTORE_TIMEOUT}" != "0" ]; then
        if gunzip -c "${ARCHIVE_FILE}" | timeout "${RESTORE_TIMEOUT}" kubectl -n "${PG_NAMESPACE}" exec -i "${PG_POD}" -- sh -lc "${KUBE_CMD}"; then
            print_status "Restore completed"
        else
            print_error "Restore failed or timed out"
            exit 1
        fi
    else
        if gunzip -c "${ARCHIVE_FILE}" | kubectl -n "${PG_NAMESPACE}" exec -i "${PG_POD}" -- sh -lc "${KUBE_CMD}"; then
            print_status "Restore completed"
        else
            print_error "Restore failed"
            exit 1
        fi
    fi

    echo ""
    echo "========================================"
    echo "Restore Complete"
    echo "========================================"
    echo "File: ${ARCHIVE_FILE}"
}

main "$@"
